// SystemUtilities.cpp -- Interface to Operating System -- 2018-09-21

// I dedicate this code to the public domain, waiving all rights 
// to the work worldwide under copyright law, including all related 
// and neighboring rights, to the extent allowed by law. You can 
// copy, modify, distribute this work, even for commercial purposes, 
// all without asking permission. 
// Don Kostuch
// October, 2018
// 

#include "stdafx.h"                // Microsoft boilerplate
#include <iostream>                // cout 
#include <fstream>                 // ifstream 
#include <iomanip>                 // setfill()  setw()
#include <sstream>                 // ostringstream ;
#include <windows.h>               // GetTempPath()  TCHAR
  using namespace std ;
#include "SystemUtilities.hpp" 
#include "Prime.hpp"               // Typedefs
#include "ostreamFork.hpp"         // Class ostreamFork

string getTempDirectory()          // Path name for tmp file
{
  TCHAR path[1001] ;               // Where name will be stored

  int len = GetTempPath( 1000 ,    // Size of buffer  
                         path ) ;  // Address of buffer for temp path name

  if ( len < 1 || len > 1000 )     // System error?
    throw runtime_error( "getTempDirectory(): temp path not found" ) ;

                                             // Convert path to std::string
  std::basic_string<TCHAR> Tpath( path ) ;
  std::string Spath( Tpath.begin(), Tpath.end() ) ;

  return Spath ;  
}

void systemError( int v , const string& cmd )    // Report system errors
{
  string m ;

  switch( v )                            // Microsoft error codes
  {
    case EPERM     : m = "Operation not permitted"   ; break  ;  //  1
    case ENOENT    : m = "No such file or directory" ; break  ;  //  2
    case ESRCH     : m = "No such process"           ; break  ;  //  3
    case EINTR     : m = "Interrupted function"      ; break  ;  //  4
    case EIO       : m = "I/O error"                 ; break  ;  //  5
    case ENXIO     : m = "No such device or address" ; break  ;  //  6
    case E2BIG	   : m = "Argument list too long"    ; break  ;  //  7
    case ENOEXEC   : m = "Exec format error"         ; break  ;  //  8
    case EBADF     : m = "Bad file number"           ; break  ;  //  9
    case ECHILD    : m = "No spawned processes"      ; break  ;  // 10
    case EAGAIN    : m = "No more processes or not enough memory or maximum nesting level reached" ; break  ; // 	11
    case ENOMEM    : m = "Not enough memory"         ; break  ;  // 12
    case EACCES    : m = "Permission denied"         ; break  ;  // 13
    case EFAULT    : m = "Bad address"               ; break  ;  // 14
    case EBUSY     : m = "Device or resource busy"   ; break  ;  // 16
    case EEXIST    : m = "File exists"               ; break  ;  // 17
    case EXDEV     : m = "Cross-device link"         ; break  ;  // 18
    case ENODEV    : m = "No such device"            ; break  ;  // 19
    case ENOTDIR   : m = "Not a directory"           ; break  ;  // 20
    case EISDIR    : m = "Is a directory"            ; break  ;  // 21
    case EINVAL    : m = "Invalid argument"          ; break  ;  // 22
    case ENFILE    : m = "Too many files open in system" ; break  ; // 23
    case EMFILE    : m = "Too many open files"       ; break  ;  // 24
    case ENOTTY    : m = "Inappropriate I/O control operation" ; break  ; // 25
    case EFBIG     : m = "File too large"            ; break  ;  // 27
    case ENOSPC    : m = "No space left on device"   ; break  ;  // 28
    case ESPIPE    : m = "Invalid seek"              ; break  ;  // 29
    case EROFS     : m = "Read-only file system"     ; break  ;  // 30
    case EMLINK    : m = "Too many links"            ; break  ;  // 31
    case EPIPE     : m = "Broken pipe"               ; break  ;  // 32
    case EDOM      : m = "Math argument"             ; break  ;  // 33
    case ERANGE    : m = "Result too large"          ; break  ;  // 34
    case EDEADLK   : m = "Resource deadlock would occur" ; break   ; // 36
    case ENAMETOOLONG : m = "Filename too long"      ; break  ;  // 38
    case ENOLCK    : m = "No locks available"        ; break  ;  // 39
    case ENOSYS    : m = "Function not supported"    ; break  ;  // 40
    case ENOTEMPTY : m = "Directory not empty"       ; break  ;  // 41
    case EILSEQ    : m = "Illegal byte sequence"     ; break  ;  // 42
    case STRUNCATE : m = "String was truncated"      ; break  ;  // 80

    default        : return ;                // No error
  }
                                           
  throw runtime_error( "system( " + cmd + " ): \n" + m ) ;
}

                                 // Execute DOS command in current process
void execute( const string& cmd , const string& errorMsg  )
{
   int systemCode = system( cmd.c_str() ) ;  // Execute DOS command 
   systemError( systemCode , cmd ) ;         // Report error, throw exception
}

void deleteFile( const string& fileName ) 
{
  remove( fileName.c_str() ) ;               // Remove from file system
} 

void setReadOnly( const string& fileName )
{
  execute( "ATTRIB +R " + fileName ) ;
}

string fileNameFromPath( const string& path )  // Name following last "\"
 {
   string fname ;

   int p = path.find_last_of( "\\" ) ;   // Rightmost "\"

   if ( p == path.size() )               // No "\" found
     fname = path ;                      // Return full stored name
   else                                  //  "\" found
     fname = path.substr( p+1 ) ;        // Extract all beyond "\"

   return fname ;
 }
                                         // Extract file extension only
 string fileExtFromPath( const string& path )
 {
   string fExt ;

   int p = path.find_last_of( "." ) ;    // Rightmost period

   if ( p == path.size() )               // No period found
     fExt = path ;                       // Return full stored name
   else                                  // Period found
     fExt = path.substr( p+1 ) ;         // Extract all beyond period

   return fExt ;
 }
  
                            // Create new binary file with all zero bytes
 void zeroBinaryFile(  fstream& file , offset length )
 {
   if ( !(file.good()) )
     throw FileError( "zeroBinaryFile() error: " ) ;

   const int size = 1024 ;   
   char zeroBuff[size]  ;              // Source of zero data

   for ( int i = 0 ; i < size ; i++ ) // Zero out source buffer
     zeroBuff[i] = '\x00' ;

   std::streampos filePosition = 0 ;  
   file.seekp( filePosition ) ;

   while ( true )                        // Write blocks of "size"
   {
     if ( (filePosition + (std::streamoff)(size)) < length )     // Room for another block?
       file.write( zeroBuff , size ) ;
     else
       break ;    
                 
     PrimeRange sz = file.tellp() ;
     filePosition += size ;
   }
                                         
   if ( length > filePosition )          // Write remaining bytes
       file.write( zeroBuff , (length-filePosition) ) ;

   if ( !(file.good()) )
     throw FileError( "zeroBinaryFile() write error: " ) ;
 
   file.seekp( 0 ) ;
   file.clear() ;
}

                                // 0 1000 becomes
                                // "000000000000_to_000000001000"
string limitsToFilename( const PrimeRange start , const PrimeRange end )
{
  ostringstream newName ;
  newName << std::setw(12) << std::setfill ('0') << start ;   // First prime
  newName << "_to_" ;
  newName << std::setw(12) << std::setfill ('0') << end ;     // Last prime
  return newName.str() ;
}

bool taskActive( string& name )        // True if program name is executing
{
  string tmpfile = getTempDirectory()
                   + "taskActiveTmp" ;  // Collects cmd output
  deleteFile( tmpfile ) ;
                                        // Compose command
  string cmd = "tasklist /NH /FI \"" + name +
               " eq " + name +  "\" > " + tmpfile ;

  execute( cmd ) ;                      // Run command

  ifstream tl( tmpfile ) ;
  if ( !tl.good() )
    throw FileError( "taskActive() file open error" ) ;

  string t ;
  getline( tl , t ) ;
  deleteFile( tmpfile ) ;

  return ( t.substr( 4 ) != "INFO" ) ;    // True if name listed
}