// Progress.hpp -- 2018-09-21
//                 Periodically release main thread progress report. 
//                 Main thread calls go() to determine when to report.
//                 go() returns true if time inteval has expired or
//                 go() has been called eventLimit times

// I dedicate this code to the public domain, waiving all rights 
// to the work worldwide under copyright law, including all related 
// and neighboring rights, to the extent allowed by law. You can 
// copy, modify, distribute this work, even for commercial purposes, 
// all without asking permission. 
// Don Kostuch
// October, 2018
// 

#ifndef PROGRESS_HPP
#define PROGRESS_HPP

#include <thread>                     // Class thread
  using namespace std ;

class Progress                        // Periodically report progress
{
private:
  volatile bool  release ;            // Permit main if() clause
  long           wait ;               // Wait time in millesconds
  long           eventCount ;         // Number of go() calls 
  long           eventLimit ;         // Release after this many calls to go()
  thread*        tp   ;               // Pointer to timer thread
public:

  Progress::Progress( double waitSeconds , long events = 0 )
  :  release( false ) ,                   // Block report initially
     wait( static_cast<long>( waitSeconds * 1000. ) )  ,   // Wait interval ms.
     eventLimit( events ) ,         // Releaese after this many calls
     eventCount( 0 )                // Number of go() calls since last release
  { 
    if ( waitSeconds > 0 )          // Start timer if interval > 0 ?
    {                               // Yes
      tp = new thread( &Progress::work , this ) ;   // Timer thread
      tp->detach() ;                // Thread becomes independent (not joinable)
      return ;
    }
     
    throw TooSmall( "Progress interval not positive: " + str( waitSeconds ) ) ;
  }

  ~Progress()                     // Destructor
  { 
    delete tp ;                   // Destroy timer thread
  } 

  void work( )                   // Runs in separate thread
  {
    while ( true )               // Do until destructor
    {                            // Sleep "wait" milliseconds
      std::this_thread::sleep_for ( std::chrono::milliseconds( wait ) ) ;
      release = true ;           // Permit main thread if() clause
    }
  }

  bool go()                    // Main thread calls to see if
  {                            //   display interval complete
    if ( release )             // Timer expired?
    {                          // Yes
      release = false ;        // Reset
      eventCount = 0 ;
      return true ;            // Release caller's if() clause
    }

    if ( ( eventLimit != 0 )  &              // Exceeded event count?
         ( eventCount >= eventLimit ) )
    {
      eventCount = 0 ;
      return true ;            // Release caller's if() clause
    }

    eventCount++ ;
    return false ;             // Not expired, no progress display
  }
} ;

#endif