// PrimeVector.cpp  -- 2018-09-24
//  List of primes to search for factors with vector functions
//  Report primality -- isPrime(v)

// I dedicate this code to the public domain, waiving all rights 
// to the work worldwide under copyright law, including all related 
// and neighboring rights, to the extent allowed by law. You can 
// copy, modify, distribute this work, even for commercial purposes, 
// all without asking permission. 
// Don Kostuch
// October, 2018
// 

#include "stdafx.h"                   // Microsoft boilerplate
#include <iostream>                   // cout
#include "Prime.hpp"                  // Typedefs, exceptions
#include "PrimeVector.hpp"            // Class PrimeVector
#include "CandidateGenerator.hpp"     // CandidateGenerator 
#include "StopWatch.hpp"              // StopWatch
#include "Progress.hpp"               // Class Progress
  using namespace std ;

                                      // Constructor
                                      // Load/build list of primes
PrimeVector::PrimeVector( const PrimeRange last ) 
: limit( last ) ,                     // Last required prime
  prog( 2.0 )                         // Load progress timer
{ 
  if ( limit > LargestPrimeRoot )
    throw Range( "PrimeVector::PrimeVector(): Limit exceeds 32 bits" ) ;

  try
  {                          // Reserve maximum number of 32 bit
                             // primes to avoid vector doubling
    reserve( 204000000 ) ;   // and likely memory fault
  }
  catch( exception ex )      // Memory fault
  {
    cout << ex.what() << endl ;
    cout << "Program requires at least 800MB" << endl ;
    pause() ;
  }
                       // Build vector from scratch
    seed() ;          // Insert the first few primes ( 2 to 29)
    extend( last ) ;  // Compute and add required primes
}


void  PrimeVector::seed( )           // Seed values
{                                    // Needed to extend() vector

  const static unsigned long sp[] =       // Seed packet
    { 2, 3, 5, 7, 11, 13, 17, 19, 23, 29 } ;

  for ( auto i = sp ; i != sp+10 ; ++i )  // Seed the vector
  {
    push_back( *i ) ;
  }
}

                                    // Extend vetor up to v
void  PrimeVector::extend( const PrimeRange v )  
{
                      // Generate prime candidates starting with 31
  CandidateGenerator cg( 31 ) ;    // 2 to 29 inserted by seed()

                             // Increase prime range 4% 
                             // to avoid nasty end of list bugs

  PrimeRange  bump( limit / 25 ) ,     // Increase amount
              extendedLimit      ;     // New, larger limit

  if ( LargestPrime - bump < limit )   // LargestPrime < limit + bump
    extendedLimit = LargestPrime ;     // Too big, use LargestPrime
  else                                 // No
    extendedLimit = limit + bump ;     // Use extended limit

  PrimeRange candidate ;               // Candidate for insertion

  cout << "Building Prime Vector"
       << endl ;

  StopWatch sw ;                   // Report time intervals
  Progress  p( 1.0 ) ;             // Report progress every 1 second
  
  do                               // Grow vector to desired size
  {
    candidate = cg.next() ;        // Get candidate (ends in 1, 3, 7, 9)

                                   // Use existing Vector elements
    if ( isPrime( candidate ) )    // to verify candidate as prime
    {
      push_back( (unsigned long)(candidate) ) ;     // It is, add it to vector
      
      if ( p.go() )                // If progress interval expired
      {
         cout << candidate         // Latest new prime
              << "   "    
              << sw.seconds()     // Total elasped time
              << "sec\r" ;  
      }
    }
  } 
  while( candidate < extendedLimit ) ;     // Any more?     

  cout << "Prime Vector complete to : "    // Progress report
       << candidate << endl ;
}
   
bool  PrimeVector::isPrime( const PrimeRange v ) 
{      
                           // Candidate generator output v
                           //   ends in 1, 3, 7, 9 and 
                           //   discards multiples of 3 and 7
                           // Search for factor starting
                           // with 11 to sqrt(v)
  const Prime lastFactor = 
      static_cast<Prime>( ceil( sqrt(v) ) ) ;  
 
  auto i    = 3 + begin() ,       // Iterator starts with 11
       endv = end() ;             // End of vector

  while( true )
  {
    if ( i == endv )              // End of vector ?
    {                             // Yes, Factor not found
      return true ;  
    }

    Prime vector_prime = *i ;    

    if  ( vector_prime > lastFactor)   // Checked up to sqrt(v)
      break ;                          // Yes, Not found, quit

    if ( 0 == v % vector_prime )       // Found factor
      return false ;                   // Yes, not prime

    i++ ;                              // Next vector prime
  }
    
  return true ;                        // No factors found
}
