// PrimeCVector.hpp -- Compact Prime Representation -- 2018-09-14

// I dedicate this code to the public domain, waiving all rights 
// to the work worldwide under copyright law, including all related 
// and neighboring rights, to the extent allowed by law. You can 
// copy, modify, distribute this work, even for commercial purposes, 
// all without asking permission. 
// Don Kostuch
// October, 2018
// 

#ifndef PRIMECVECTOR_HPP
#define PRIMECVECTOR_HPP

#include "Prime.hpp"                  // typedefs

typedef unsigned char FlagByte ;      // Unsigned to avoid sign shifts

// PrimeCVector takes advantage of all primes 
// (except single digit) end in 1, 3, 7, or 9.

// The primality of values less than 23 (which are not stored)
// are handled by special case code.
// Primality tests involving small values provide
// a performance advantage.

// Values 20 or greater are stored in groups of 20
// with each group represented in one byte, as in 
// this sample from 20 to 59.

// Byte 0 is retained as 0x00 but is ignored, 
// and is handled by special case code.

// Byte    1                       |  2
// Bit     0  1  2  3  4  5  6  7  |  0  1  2  3  4  5  6  7
// Value  21 23 27 29 31 33 37 39  | 41 43 47 49 51 53 57 59
// Prime?  F  T  F  T  T  F  T  F  |  T  T  T  F  F  T  F  T
// Hex     0X5A                    |  0XE5

// PrimeCVector provides the common logic for encoding and decoding
// PrimeC data. It is abstract. Derived classes must provide the
// the function getByte() and putByte() to access their data representation.
//
// These classes include:
//   Class Name         Representation   Purpose
//   ----------------   --------------   -----------------------------------
//   PrimeCArray        char array       Performance for creation and access
//   PrimeCFileWriter   Disk file        Create persistant PrimeC data file
//   PrimeCFileReader   Disk file        Access PrimeC disk data file
//
//  Much of PrimeCVector in inline for performance

class PrimeCVector               // Encode/decode PrimeC data
{
protected:                       // FirstValue and lastValue are multiples of 20
  PrimeRange firstValue ,        // Beginning of range of primes  
             lastValue  ,        // End of range of primes  
             len        ,        // Byte count
             sum        ;        // Redundancy check
  
                                 // Range exception if not in range
                                 // Return true if small (v<23)
  bool inRange( const PrimeRange v ) const 
  {
    switch( v )                  // Special case
    {
      case  2:                   // Primes less than 23
      case  3:
      case  5:
      case  7:
      case 11:
      case 13:
      case 17:
      case 19: return true ;     // Is small prime
    } ;                          // and in range

    if ( v < 23 )                // Small but not prime
      throw Range( "Value not prime: " , v ) ;   
                   
                                      // v >= 23                          
    if ( ( 0 == (v & 0X01) ) |        // Even?
         ( 0 == (v % 5 )   ) |        // Multiple of 5?
         ( v < firstValue  ) |        // Not in range of PrimeCVector
         ( lastValue < v   ) )
      throw Range( "Value outside range, 5*n, or even : " , v ) ;

    return false ;               // Ordinary (not small) case
  }
  
                       // Byte location of v representation
  offset byteOffset( const PrimeRange v ) const ;

                       // Return bit location of v representation
  FlagByte mask( const PrimeRange v ) const ;

public:  
  PrimeCVector( )                      // Default constructor
  : firstValue(0) ,   
    lastValue(0) , 
    len(0) , 
    sum(0)  
  {}
                                       // Constructor
  PrimeCVector( PrimeRange first ,     // Initial range
                PrimeRange last  ) ;

  void init( PrimeRange first ,        // Replacement range
             PrimeRange last  ) ;

  PrimeRange first( ) const            // First value in this series
  {
    return firstValue ; 
  }

  PrimeRange last( ) const             // Last nalue in this series
  {
    return lastValue ; 
  }

  PrimeRange length() const            // Bytes in vector
  {  
    return len ;
  }

  PrimeRange chk() const        // Check value, sum of all bytes
  {  
    return sum ;
  }

  class iterator ;              // Location bookeeping; forward declaration

                                // Location of v, or exception if v not prime
  PrimeCVector::iterator find( const PrimeRange v ) ;

                                // Return v or next value ending in 1,3,7,9
  Prime                  nextPossible( const PrimeRange v ) ;

                                // Return iterator to v or subsequent prime.
  PrimeCVector::iterator findNextPrime( const PrimeRange v ) ;

  bool isPrime( const PrimeRange v )  ;

  void     set( const Prime v ) ;   // Set primes flag for v

  void computeCheck() ;             // Compute and store "sum"

  void verifyCheck()  ;             // Verify file data
    
                                    // Subclass access methods
                                    // Must be defined in subclass
 
  virtual FlagByte getByte( const offset off ) = 0 ; 

  virtual void     putByte( const offset off , const FlagByte c ) = 0 ;
} ;

///////////////////////////////////////

class PrimeCVector::iterator      // Location bookeeping
{
private:
  PrimeCVector*   pcv ;            // Parent container (Array or file)
  Prime           pv ;             // Prime value referred to
  offset          byteLocation ;   // Byte position of pv's flag

                                   // unsigned to avoid sign shift
  FlagByte        bitmask ;        // Where pv's flag is stored
  FlagByte        primeFlags ;     // Cache of File flag byte at byteLocation

  int             valid   ;        // 0 == OK, -1 == too small, +1 == too big

public:
  PrimeCVector::iterator( )      // Default constructor
  : pcv(0)          ,            // Parent container pointer
    pv(0)           ,            // Prime value referred to
    byteLocation(0) ,            // Byte position of prime's flag
    bitmask(0)      ,            // Bit position of prime
    primeFlags(0)   ,            // File flag byte at byteLocation
    valid(0)                     // Iteraror OK
  {}

  PrimeCVector::iterator( PrimeCVector* t ,   // Parent container
                          const Prime v ) ;   // Prime value

                                  // Throws exception on error
  Prime  operator*() ;            // Return current prime target
                                  //  stored in iterator
   
  void   operator++( int ) ;      // Move to next prime and cache it

  void   operator--( int ) ;      // Move to previous prime and cache it
} ;

#endif