// PrimeCVector.cpp -- Process Compact Prime Representation -- 2018-09-14

// I dedicate this code to the public domain, waiving all rights 
// to the work worldwide under copyright law, including all related 
// and neighboring rights, to the extent allowed by law. You can 
// copy, modify, distribute this work, even for commercial purposes, 
// all without asking permission. 
// Don Kostuch
// October, 2018
// 

#include "stdafx.h"                    // Microsoft boilerplate
#include "PrimeCVector.hpp"            // Class PrimeCVector

                                       // Special case for small values
static const bool smallPrime[23] = 
//  0 1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16 17 18 19 20 21 22
  { 0,0,1,1,0,1,0,1,0,0, 0, 1, 0, 1, 0, 0, 0, 1, 0, 1, 0, 0, 0 } ;

void PrimeCVector::init( const PrimeRange first ,                           
                         const PrimeRange last ) 
{ 
  firstValue = 20 * ( first / 20 ) ;   // Round down to multiple of 20
  lastValue  = 20 * ( last  / 20 )  ; 

  if (  (first >= last) )
    throw Range( "PrimeCVector::init(): First not less than last" ) ;
  len = (last - first) / 20  ;      
  sum = 0 ;                               // Check sum
}

                                                        // Constructor 
PrimeCVector::PrimeCVector( const PrimeRange first ,                           
                            const PrimeRange last ) 
{ 
  init( first , last ) ;                           
}

                                          // Locatation of FlagByte for v
offset PrimeCVector::byteOffset( const PrimeRange v ) const 
{
  return (offset)(( v - firstValue )  / 20 ) ;
}

 
                           // Bit location of v representation in FlagByte
FlagByte PrimeCVector::mask( const PrimeRange v ) const
{    
  switch ( ( v - firstValue )  % 20  )    // Convert to bit position
  {
    case  1:  return '\x80' ;      // 1000 0000
    case  3:  return '\x40' ;      // 0100 0000
    case  7:  return '\x20' ;      // 0010 0000
    case  9:  return '\x10' ;      // 0001 0000

    case  11: return '\x08' ;      // 0000 1000
    case  13: return '\x04' ;      // 0000 0100
    case  17: return '\x02' ;      // 0000 0010
    case  19: return '\x01' ;      // 0000 0001
  
    default : throw Range( "PrimeCVector::mask() error : " , v ) ;
  } ;
}

void PrimeCVector::computeCheck()          // Check value of PrimeC list
{
  sum = 0 ;

  for ( offset i(0) ; i < length() ; i++ )
  {
    FlagByte c = getByte( i ) ;
    sum += c ;
  }
}

void PrimeCVector::verifyCheck()   // Computed check == stored check?
{
  PrimeRange s( 0 ) ;

  for ( offset i(0) ; i < length() ; i++ )
  {
    FlagByte c = getByte( i ) ;
    s += c ;
  }

  if ( sum == s ) 
    return ;

  throw PrimeException( "PrimeCVector::verifyCheck() error" ) ;
}

                   // Return next possible prime above v
Prime PrimeCVector::nextPossible( const PrimeRange v )
{
  PrimeRange r ;

  if ( v < 10 )                 // For small primes
  {
    switch( v ) 
    {
      case 0:   
      case 1:  r = 2 ;  break ;

      case 2:  r = 3 ;  break ;

      case 3:
      case 4:  r = 5 ;  break ;
                   
      case 5:
      case 6:  r = 7 ;  break ;

      case 7:   
      case 8:  r = 9 ;  break ;

      case 9:  r = 11 ;  break ;
    }
  }
  else                       // v > 10
  {
    switch( v % 10 )         // Units digit
    {                   // New units digit  
      case 0:           // 1
      case 2:           // 3
      case 6:           // 7
      case 8:           // 9 
        r =  v + 1 ;  
        break ;

      case 1:           // 3
      case 5:           // 7
      case 7:           // 9
      case 9:           // 1  (11)
        r = v + 2 ;  
        break ;

      case 3:           // 7
        r = v + 4 ;  
        break ;
    }
  }

  if ( r > last() )
    throw Range( "PrimeCVector::nextPossible() " , v ) ;

  return r ;
}

PrimeCVector::iterator  PrimeCVector::findNextPrime( const PrimeRange v ) 
{
  PrimeRange vtest = nextPossible( v ) ;

  while( true )                    // Test candidates until prime
  {
    if ( isPrime( vtest ) )        // Hit on file bit mask?
      return PrimeCVector::iterator( this , vtest ) ;

    vtest = nextPossible( vtest ) ; // No, try the next one
  }
}

                               // Return location of v; v must be prime
PrimeCVector::iterator PrimeCVector::find( const PrimeRange v )  
{
  return PrimeCVector::iterator( this , v ) ;
}

PrimeCVector::iterator::iterator( PrimeCVector* t ,    // PrimeC object pointer
                                  const Prime   v )    // Value pointed to
{
    valid = 0 ;                       // OK
    bool small = t->inRange( v ) ;    // Throw Range (even or out of bounds)
    pv  = v ;                         // Target value
    pcv = t ;                         // Pointer to PrimeC object
    if ( small )                      // Less than 23
      return ;                        // Not in file, skip rest

    byteLocation = t->byteOffset( v ) ;          // Offset of primeFlags
    bitmask      = t->mask( v ) ;                // Bit position of v in primeFlags
    primeFlags   = t->getByte( byteLocation ) ;  // File byte containg v

    if ( ! ( primeFlags & bitmask )  )           // Not prime?
      throw Range( "PrimeCVector::iterator::iterator, not prime " , v ) ;
}

 Prime  PrimeCVector::iterator::operator*()      // Return prime pointed to
 {                                               // from cache in iterator
   if ( 0 != valid )                             // Not too big or small
     throw EndOfFile( "PrimeCVector::iterator::operator*()" );
   return pv ; 
 }    // Return current prime target


void PrimeCVector::iterator::operator++(int)      // Move to next prime
{
  if ( pv >= pcv->last() )                        // Already at EOF?
    throw EndOfFile( "PrimeCVector::iterator::operator++(): " + str(pv) ) ;           // Yes

  if ( pv < 23 )                                  // Special cases for 0 to 22
  {
    switch ( pv )
    {
      case   2: pv =   3 ; return ;
      case   3: pv =   5 ; return ;
      case   5: pv =   7 ; return ;
      case   7: pv =  11 ; return ;
      case  11: pv =  13 ; return ;
      case  13: pv =  17 ; return ;
      case  17: pv =  19 ; return ;

      case  19: pv           =  23 ; 
                byteLocation =   1 ;
                bitmask      = '\x40' ;
                primeFlags   = pcv->getByte( byteLocation ) ;
                return ;

      default : throw Range( "PrimeCVector::iterator::operator++(): " + str( pv ) ) ;
    }  // End switch
  }    // End if ( pv <= 23 ) 

                             // End special case, regular processing
  PrimeRange newpv = pv ;

  do 
  {                          // Move to next possible position 
    switch ( bitmask )       // Based on current slot
    {
      case 0x0080u :                 //  1 to  3
      case 0x0020u :                 //  7 to  9
      case 0x0010u :                 //  9 to 11
      case 0x0008u :                 // 11 to 13
      case 0x0002u :                 // 17 to 19
        newpv += 2 ;                 // Increment prime value
        bitmask >>= 1 ;              // Next bit position
        break ;

      case 0x0040u :                 //  3 to  7     
      case 0x0004u :                 // 13 to 17
        newpv += 4 ;                 // Increment prime value
        bitmask >>= 1 ;              // Next bit position
        break ;

      case 0x0001u :                 // 19 to 1 in next byte
        newpv += 2 ;                 // Increment prime value
        byteLocation++ ;             // Next file byte 
        primeFlags = pcv->getByte( byteLocation ) ;
        bitmask = '\x80' ;           // First value in byte
        break ; 
    } ;

    if ( newpv == pcv->last() )         // Last prime in file?
      break ;                           // Yes, valid 
 
    if ( newpv > pcv->last() )          // Past end of file
    {
      valid = 1 ;                       // Beyond last prime
      return ;
    }
  }
  while ( !(bitmask & primeFlags ) ) ;    // Continue if Not prime

   pv = newpv ;                           // Found next prime
 }

void PrimeCVector::iterator::operator--(int) 
{
  if ( pv <= pcv->first() )
    throw EndOfFile( "PrimeCVector::iterator::operator--(): " + str(pv) ) ;

  if ( pv <= 23 )                     // Special cases for 0 to 23
  {
    switch ( pv )
    {
      case  23: pv = 19 ; return ;
      case  19: pv = 17 ; return ;
      case  17: pv = 13 ; return ;
      case  13: pv = 11 ; return ;
      case  11: pv =  7 ; return ;
      case   7: pv =  5 ; return ;
      case   5: pv =  3 ; return ;
      case   3: pv =  2 ; return ;
      case   2: throw EndOfFile( "PrimeCVector::iterator::operator--(): "+ str( pv )  ) ; ;

      default : throw Range( "PrimeCVector::iterator::operator--() error: " + str( pv ) ) ;
    }
  }

                             // End special case, regular processing
  PrimeRange newpv = pv ;

  do                         // Search for previous prime
  {                          // Move to previous possible position 
    switch ( bitmask )       // Based on current slot
    {
      case '\x80' :                 // Move from 1 to previous 19
        newpv -= 2 ;                // Decrement prime value
        byteLocation-- ;            // Previous file byte 
        primeFlags = pcv->getByte( byteLocation ) ;
        bitmask = '\x01' ;          // Last value in byte (19 flag)
        break ; 

      case 0x0040u :                 //  3 to  1 
      case 0x0010u :                 //  9 to  7 
      case 0x0008u :                 // 11 to  9
      case 0x0004u :                 // 13 to 11
      case 0x0001u:                  // 19 to 17
        newpv -= 2 ;                    // Decrement prime value
        bitmask <<= 1 ;              // Previous bit position
        break ; 

      case 0x0020u :                 //  7 to  3        
      case 0x0002u :                 // 17 to 13
        newpv -= 4 ;                 // Decrement prime value 
        bitmask <<= 1 ;              // Previous bit position
        break ;
    } 

    if ( pv == pcv->first() )
      break ;
    if ( pv < pcv->first() )
    {
      valid = -1 ;                  // Too small
      return ;
    }
  }
  while ( !( bitmask & primeFlags ) ) ;     // Continue while not prime

  pv = newpv ;                         // Found previous prime
}

bool  PrimeCVector::isPrime( const PrimeRange v )  
{ 
  if ( v < 23 )             // Special case
    return smallPrime[v] ;

  if ( ( 0 == (v & 0x01) ) |        // Even or
       ( 0 == (v % 5   ) ) )        // divissible by 5
    return false ;

  offset byteLocation = byteOffset( v ) ;   // Flag  byte location in file/array
  FlagByte c  = getByte( byteLocation ) ;   
  return  0 != ( c & mask( v ) ) ;          // Mask/flag match? Yes==prime
}

void  PrimeCVector::set( const Prime v )    // Set flag bit for v
{
  if ( v < 23 )             // Special case
    return ;                // Handled in ++ -- and isPrime()

  offset byteLocation = byteOffset( v ) ;
  FlagByte c  = getByte( byteLocation ) ;  // Get current byte
  c |= mask( v ) ;                     // Insert new bit (value)
  putByte( byteLocation , c ) ;        // Put new byte
}
