// PrimeCFileWriter.cpp -- Interface to primec (Compact) prime files.
//                          2019-09-14

// I dedicate this code to the public domain, waiving all rights 
// to the work worldwide under copyright law, including all related 
// and neighboring rights, to the extent allowed by law. You can 
// copy, modify, distribute this work, even for commercial purposes, 
// all without asking permission. 
// Don Kostuch
// October, 2018
// 

#include "stdafx.h"                // Microsoft Boilerplate
#include "PrimeCFileWriter.hpp"    // Class PrimeCFileWriter
#include "SystemUtilities.hpp"     // setReadOnly()  limitsToFilename()
#include "Progress.hpp"            // Class Progress
#include "OstreamFork.hpp"         // cout_cerr
#include <numeric>                 // accumulate()
  using namespace std ;

const static PrimeRange                        // 24 bytes of metadata
  headerLength( 3 * sizeof(PrimeRange) ) ;

PrimeCFileWriter::PrimeCFileWriter( PrimeRange first , PrimeRange last )
: PrimeCVector( first , last )         // Base class init 
{ 
  if ( length() >= SIZE_MAX )          // 2^32
    throw TooBig( "PrimeCFileWriter::PrimeCFileWriter() file size too big" ) ;

  fileName = limitsToFilename( first, last ) ;   // first_to_last (12 digits)
  fileName += ".primec" ;                                    // File type
  
                        // Metadata     margin     data                         
  offset fileLength( headerLength + 10 + ( (last-first)/20 ) ) ;

  cout_cerr << "Initializing PrimeC file " << fileName << endl ;
  
  cFile.open( fileName , ios::in + ios::out + ios:: binary + ios::trunc ) ;
  if (!cFile.good())
    throw FileError( "PrimeCFileWriter::PrimeCFileWriter(): File error" ) ;

  zeroBinaryFile( cFile , fileLength ) ;

  PrimeRange fst( first ) ,
             lst( last  ) ,
             chk( 0 )     ;

                                  // Write first, last; destructor writes chk
  cFile.seekp( 0 ) ;
  cFile.write( (char*)(&fst) , sizeof(fst) ) ;  // Start of series
  cFile.write( (char*)(&lst) , sizeof(lst) ) ;  // End of series
  cFile.write( (char*)(&chk) , sizeof(chk) ) ;  // Check value (filled by destructor)
  if (!cFile.good())
    throw FileError( "PrimeCFileWriter::PrimeCFileWriter(): File error" ) ;
}

PrimeRange PrimeCFileWriter::computeSum() // Value of PrimeC list check sum
{                                         // Overload for performance
  PrimeRange total( 0 ) ;
  const offset bufLen( 10240 ) ;
  char buf[ bufLen ] ;
  cFile.seekg( headerLength ) ;           // Start of data (past metadata)
  offset i( 0 ) ;

  do                            // Compute in 10240 byte chunks
  {                             // On last read (EOF) gcount() indicate actual number
    if ( !(cFile.good()) )
      throw FileError( " PrimeCFileWriter::computeCheck() error" ) ;
  
    cFile.read( buf , bufLen ) ; 
    total = std::accumulate( buf , buf + cFile.gcount() , total ) ;
  }
  while( !(cFile.eof()) ) ;

  return total ;
}

void PrimeCFileWriter::computeCheck()     // Store check value of PrimeC list
{                                         // Overload for performance
  sum = computeSum() ;
}

void PrimeCFileWriter::verifyCheck()      // Performance overload
{
  if ( sum == computeSum() ) 
    return ;

  throw PrimeException( "PrimeCArray::verifyCheck() error" ) ;
}

PrimeCFileWriter::~PrimeCFileWriter( )  // Write check sum snf close file
{
  computeCheck() ;                      // Set sum in PrimeCVector
  PrimeRange chk = sum ;                // Write check sum to file
  cFile.seekp( 16 ) ;                   // Check sum location
  cFile.write( (char*)(&chk) , sizeof(chk) ) ;    // Check value
  cFile.close() ;
  cout_cerr << "PrimeCFile complete: "
            << fileName << endl ;
  setReadOnly( fileName ) ;
}

Prime PrimeCFileWriter::getPrime() 
{ 
  Prime v = *getIteratorP ;        // Return current iterator cached target
  getIteratorP++ ;                 // Move to next prime
  return v ;
}

                                   // Fetch flagbyte from file
FlagByte PrimeCFileWriter::getByte( const offset off ) 
{
  static FlagByte c ;
  cFile.seekg( off + headerLength ) ;
  cFile.read( (char*)(&c) , 1 ) ;
  if ( ! cFile.good() )
     throw FileError( "PrimeCFileWriter::getByte() file read error: " + fileName ) ;
  return c ;
}
                                    // Write flagbyte to file
void PrimeCFileWriter::putByte( const offset off , 
                                const FlagByte c ) 
{
  cFile.seekp( off + headerLength ) ;
  cFile.write( (char*)(&c) , 1 ) ;
  if ( ! cFile.good() )
    throw FileError( "PrimeCFileWriter::putByte() file write error: "  + fileName ) ;
}

                                // Copy PrimeCArrray to file
 void PrimeCFileWriter::operator=( PrimeCArray& pca )
 {
   if ( ( first() != pca.first() ) |
        ( last()  != pca.last()  ) )
     throw PrimeException( "PrimeCFile::operator=() : Not same range: " 
                            + fileName ) ;
  
                                // Write check value and data
  PrimeRange c = pca.chk() ;
  cFile.seekp( 16 ) ;
  cFile.write( (char*)(&c) , sizeof(c) ) ;  // Check value 
  cFile.write( (char*)(pca.arrayPointer()) , pca.length() ) ;
  if ( !cFile.good() )
    throw FileError( "PrimeCFile::operator=() : Write error "
                          + fileName ) ;
}