// TxtToPrimeC.cpp : Converts Prime txt files to primec.
// A primec file stores primes as groups of 20 within one byte.
// Each bit represents the primality of the 8 possible candidates
// in that range, ie., 21, 23, 27, 29, 31, 33, 37, 39.
// Numbers ending in 0, 2, 4, 5, 6, 8 are not prime (except for 2 and 5).
// The range of values in a file is defined by the file name, ie. 
//   3_to_1000.primec

// I dedicate this code to the public domain, waiving all rights 
// to the work worldwide under copyright law, including all related 
// and neighboring rights, to the extent allowed by law. You can 
// copy, modify, distribute this work, even for commercial purposes, 
// all without asking permission. 
// Don Kostuch
// October, 2018
// 

#include "stdafx.h"                // Microsoft Boilerplate
#include <string>                  // Class string
#include <fstream>                 // Class fstream
#include <iomanip>                 // setprecision()
  using namespace std ;
#include "Prime.hpp"               // Typedef, inline utilities, exceptions
#include "DirectoryStream.hpp"     // Class DirectoryStream
#include "Progress.hpp"            // Class Progress (report timer)
#include "StopWatch.hpp"           // Class StopWatch
#include "PrimeCFileWriter.hpp"    // Class PrimeCFileWriter
#include "OstreamFork.hpp"         // Class ostreamFork, cout_cerr

const static PrimeRange fileRange( 10000000000 ) ; // Ten Billion
//const static PrimeRange fileRange( 10000 ) ;


int main( int argc , char* argv[] )
{
  if ( argc != 4 )
  {
    cout << "TxtDirectory start end\n"
         << "start >=2 , end > start" << endl ;
    return 1 ;
  }

  string directoryName( argv[1] ) ,
         startTxt(      argv[2] ) ,
         endTxt(        argv[3] ) ;

  ofstream logFile( startTxt + "Log.txt" ) ;
  cout_cerr.setStream(  cout , logFile ) ;
  showCommas( cout ) ;                    // Show comma separators
  showCommas( logFile ) ;                 //  in integers

  PrimeRange start ,
             end   ,
             current_start ,
             endFile ;
  double     range ; 

  try
  {
    start         = stoull( startTxt ) ;
    end           = stoull( endTxt   ) ;
    current_start = start ;
    endFile       = current_start + fileRange ;
    range         = static_cast<double>(end - start) ;
  }
  catch( exception ex )
  {
    cout_cerr << "TxtToPrimeC: Start or end conversion error." << endl ;
    return 2 ;
  }

  DirectoryStream<Prime> primeStream( directoryName ) ;
  Prime     v = 0 ;
  StopWatch sw ;
  Progress p( 10.0 , 0 ) ;       // 10 second report cycle

  PrimeCFileWriter*  p_pcf = 0 ; 
  PrimeCArray        pca( current_start , endFile ) ; 

  try
  {
    do                          // Skip files to start
    {
      v = primeStream.get() ;   // Get a txt prime

      if ( (v < start) & p.go() )
        cout_cerr  << "Skipping: " << v << endl ;

      if ( start > ( v + 300000000 ) )    // More than one file
       primeStream.next() ;         
    }
    while ( v < start ) ;          // Skip to start

    while ( true )                 // Process several output files
    {                              // File name = start_to_endFile.primec
      p_pcf = new PrimeCFileWriter( current_start , endFile ) ;  

      while ( true )                  // v is in range
      {
        if ( v > end ) 
          throw EndOfFile( "End of primec range: " +str(v) ) ;

        if ( v > endFile )            // End current file?
        {                             // Yes
          pca.computeCheck() ;
          *p_pcf = pca ;              // Copy array to file
          cout_cerr << "TxtToPrimeC file done: " << endl 
                    << current_start << " to " << endFile << endl ; 
            
          delete p_pcf ;              // Writes file to disk
          current_start = endFile ;
          endFile       = current_start + fileRange ;
          p_pcf = new PrimeCFileWriter( current_start , endFile ) ;   
          pca.newRange( current_start , endFile ) ;  // Restart pca
        }  // End  if ( v > endFile )  
           
        pca.set( v ) ;             // Store v in primec array

        if ( p.go() )                 // Time for progress report?
        {
          double elapsed = sw.seconds() ,
                 total   = elapsed * range / (v-start) ,
                 remaining = total - elapsed ; 
          cout_cerr << setprecision( 4 ) << v 
                    << " Elapsed: "   << elapsed
                    << " Remaining: " << remaining
                    << "  " << 100. * elapsed/total
                    << "%"  << endl ;
        }  // End if ( p.go() )

        v = primeStream.get() ;        // Get a txt prime

      } // End while ( true )   // v is in range

      } //  while ( true )      // Process several output files
    }
    catch( EndOfFile pe )             // End of txt input
    {
       pca.computeCheck() ;
       *p_pcf = pca ;              // Copy array to file
       cout_cerr << "TxtToPrimeC file done: " << endl 
                    << current_start << " to " << endFile << endl ; 
  
       delete p_pcf ;              // Writes file to disk
       cout_cerr << "End of Txt input" << endl ;
       return 0 ;
    }
    catch( PrimeException ex )
    {
      cout_cerr  << ex.what() << endl ;
      cout_cerr  << "TxtToPrimeC PrimeException exit" << endl ; 
      return 1 ;
    }
    catch( exception ex )
    {
      cout_cerr  << ex.what() << endl ;
      cout_cerr  << "TxtToPrimeC exception exit" << endl ; 
      return 1 ;
    }
 
  cout_cerr  << "TxtToPrimeC Abnormal exit" << endl ;
	return 2 ;
}

