// TxtToPrimeC.cpp : Converts Prime txt files to primec.
// A primec file stores primes as groups of 20 within one byte.
// Each bit represents the primality of the 8 possible candidates
// in that range, ie., 21, 23, 27, 29, 31, 33, 37, 39.
// Numbers ending in 0, 2, 4, 5, 6, 8 are not prime (except for 2 and 5).
// The range of values in a file is defined by the file name, ie. 
//   3_to_1000.primec

// I dedicate this code to the public domain, waiving all rights 
// to the work worldwide under copyright law, including all related 
// and neighboring rights, to the extent allowed by law. You can 
// copy, modify, distribute this work, even for commercial purposes, 
// all without asking permission. 
// Don Kostuch
// October, 2018
// 

#include "stdafx.h"                // Microsoft Boilerplate
#include <string>                  // Class string
#include <fstream>                 // Class fstream
#include <iomanip>                 // setprecision()
  using namespace std ;
#include "Prime.hpp"               // Typedef, inline utilities, exceptions
#include "DirectoryStream.hpp"     // Class DirectoryStream
#include "Progress.hpp"            // Class Progress (report timer)
#include "StopWatch.hpp"           // Class StopWatch
#include "PrimeCFileWriter.hpp"    // Class PrimeCFileWriter
#include "OstreamFork.hpp"         // Class ostreamFork, cout_cerr

int main( int argc , char* argv[] )
{
  showCommas( cout ) ;                 // Show comma separators
  showCommas( cerr ) ;                 //  in integers

  if ( argc != 4 )
  {
    cout_cerr << "TxtDirectory start end\n"
              << "start >=2 , end > start" << endl ;
    return 1 ;
  }

  string directoryName( argv[1] ) ,
         startTxt(      argv[2] ) ,
         endTxt(        argv[3] ) ;

  PrimeRange start ,
             end   ;
  double     range ; 

  try
  {
    start = stoull( startTxt ) ;
    end   = stoull( endTxt   ) ;
    range = static_cast<double>(end - start) ;
  }
  catch( exception ex )
  {
    cout_cerr << "TxtToPrimeC: Start or end conversion error." << endl ;
    return 2 ;
  }
  
  PrimeCFileWriter  pcf( start , end ) ;   // File name = start_to_end.primec
  PrimeCArray       pca( start , end ) ;  

  try
  {
    DirectoryStream<Prime> primeStream( directoryName ) ;

    Prime    v = 0 ;
    StopWatch sw ;
    Progress p( 5.0 , 0 ) ;     // 5 second report cycle

    do                          // Skip files to start
    {
      v = primeStream.get() ;   // Get a txt prime

      if ( p.go() )
        cout_cerr  << "Skipping: " << v << endl ;

      if ( start > ( v + 300000000 ) )
        primeStream.next() ;         
    }
    while ( v < start ) ;        // Skip to start

    while ( true )               // v is in renage
    {
      if ( v > end ) 
        throw EndOfFile( "End of primec range: " +str(v) ) ;
           
      pca.set( v ) ;             // Store v in primec array

      if ( p.go() )
      {
        double elapsed = sw.seconds() ,
               total   = elapsed * range / (v-start) ,
               remaining = total - elapsed ; 
        cout_cerr << setprecision( 3 ) << v 
                  << " Elapsed: "   << elapsed
                  << " Remaining: " << remaining
                  << "  " << 100. * elapsed/total
                  << "%"  << endl ;
      }

      v = primeStream.get() ;        // Get a txt prime
    }
  }
  catch( EndOfFile pe )              // Not error
  {                                  // Copy Array to file
    pca.computeCheck() ;
    pcf = pca ;  
    cout_cerr << "TxtToPrimeC normal EOF exit" << endl 
              << startTxt << " to " << endTxt << endl ; 
    return 0 ;
  }
  catch( exception ex )
  {
    cout_cerr  << ex.what() << endl ;
    cout_cerr  << "TxtToPrimeC exception exit" << endl ; 
    return 1 ;
  }
 
  cout_cerr  << "TxtToPrimeC Abnormal exit" << endl ;
	return 2 ;
}

