// Goldbach.cpp : Test Goldbach Conjecture:  2018-09-25
//                Any even number greater than 2
//                is the sum of 2 primes

// I dedicate this code to the public domain, waiving all rights 
// to the work worldwide under copyright law, including all related 
// and neighboring rights, to the extent allowed by law. You can 
// copy, modify, distribute this work, even for commercial purposes, 
// all without asking permission. 
// Don Kostuch
// October, 2018
// 

#include "stdafx.h"               // Microsoft Boilerplate
#include "Prime.hpp"              // typedefs, exceptions
#include "PrimeCArray.hpp"        // Class PrimeCArray
#include "PrimeFloatVector.hpp"   // Floating cache
#include "Progress.hpp"           // Gate for periodic reports
#include "StopWatch.hpp"          // Timer
#include <string>                 // string, stoull()
#include <iomanip>                // setprecision()
  using namespace std ;

void errorExit( int exitCode , string detail = "" )
{
  cout << "\nGoldBach firstEven lastEven smallPC floatPC\n" 
       <<   "         int64     int64    file    file\n" 
       << detail 
       << endl ;
  pause() ;
  exit( exitCode ) ;
}

int main( int argc , char* argv[] )
{
         // Command line parameter collection and checking

  if ( argc != 5 )
    errorExit( 1 ) ;
  
  string firstGBs( argv[1] ) ,   // First Goldbach even
         lastGBs(  argv[2] ) ,   // Last Goldbach even
         spCName(  argv[3] ) ,   // Small primes PrimeC file
         fpName (  argv[4] ) ;   // PrimeC directory

  PrimeRange firstGB ,           // First Goldbach even
             lastGB  ;           // Last Goldbach even
  try
  {
    firstGB = stoull( firstGBs ) ,  // First Goldbach even
    lastGB  = stoull( lastGBs  ) ;  // Last Goldbach even
  }
  catch( exception ex )             // Catch conversion error
  {
    errorExit( 2 , "Numeric conversion error of firstEven or lastEven " )  ;
  }

  firstGB = ( firstGB/ 2 ) * 2 ;    // Round down to even
  lastGB  = ( lastGB / 2 ) * 2 ;

                                   // Command parameter check
  if ( (firstGB < 4) | (firstGB > lastGB) )
    errorExit( 3 , "Even Range error" ) ;
 
  showCommas() ;               // Display integers with commas
  Progress p( 2.0 ) ;          // Progress display timer
  StopWatch sw ;               // Interval timer
  cout << setprecision( 3 ) ;

  double range = (double)(lastGB - firstGB ); // For progress report

                               // Announce start of process
  cout << "Goldbach pair check of range :\n  " 
       << firstGB << " to " << lastGB << endl ;

  PrimeRange even ;

  try
  {
    PrimeCFileReader pcfr( spCName ) ;
    PrimeCArray      pcSmall ( pcfr ) ;       // Small primes list

    PrimeFloatVector pcFloat ( fpName , firstGB ) ; // Floating prime list
    pcSmall.verifyCheck() ; 

    cout << "\nProcessing Goldbach pairs" << endl ;

    Prime diff ;                         // Even - first prime
                                         // Examine range of evens
                                            
    for ( even = firstGB ; even <= lastGB ; even+=2 )  // Range of evens
    {
    if ( p.go() )                        // Progress message
      {
        PrimeRange done = even - firstGB ;
        double     pct  = 100. * ( done / range ) ;

        cout << even        << " : "          // Cueent even
             << sw.hours()  << " hr.  "       // Elapsed hours
             << pct         << " %     \r" ;  // Percent done
      }          
                         
                        // Search for Goldbach pair (even = prime1 + prime2)
      Prime addend = even ;
      vpi   addend_itr =                 // Find previous prime
		    pcFloat.previous( addend ) ; 

      try
      {
        while ( true )
        { 
          addend = *addend_itr ;          // Get prime addend    
          diff   = even - addend ;   
          if ( pcSmall.isPrime( diff ) )  // Pair found ?
            break ;                       // Yes, next even

                                          // Out of addends?
          if ( pcFloat.begin() == addend_itr )
		      {                               // Yes
            throw PrimeException( "Goldbach: Prime pair not found" ) ;
		      }

          addend_itr-- ;                  // No, try previous prime
        } 
        
      }
      catch( EndOfFile ex )
      {
         cout << "\nPrime addends not found for " 
              << even << endl ;
      }
     
    }    // End for     pair search
  }      // End try

  catch( EndOfFile ex )    // Last Goldbach beyond file range
  {
    string msg = "End of file\n" + str( even )
                 + " and successors not processed \n" ;

    errorExit( 4 , msg ) ;    
  }

  catch( exception ex )    // Unexpected exception 
  {                        // Program or system error 

    string msg = "\nUnexpected exception" ;
           msg += ex.what() ;

    errorExit( 5 , msg ) ;    
  }
                              // That's all folks
  cout << "\nNormal end\n" 
       << even-2 << " processed\n" << endl ;
  pause() ;
	return 0 ;
}

